
suppressMessages(library(RKernel))
options(width=120)

# ## Preparing the data
# The data we are using are in the file "ESS-relig-CHES.RData".

suppressMessages(library(memisc))
library(magrittr)


load(file="ESS-relig-CHES.RData")


load(file="perc-denom-country-wide.RData")


ESS_work_west <- subset(ESS_pray_denom_CHES,
    country2 %in% c(
        "Austria",
        "Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels",
        "Denmark",
        "France",
        "Germany (West)","Germany/Bavaria",
        "Ireland",
        "Italy",
        "Luxembourg",
        "Netherlands",
        "Norway",
        "Portugal",
        "Spain/Castilia","Spain/Catalunia","Spain/Basque",
        "Sweden",
        "Switzerland",
        "United Kingdom/England",
        "United Kingdom/Scotland",
        "United Kingdom/Wales"
    ) & !is.na(sociallifestyle),
    select=c(CHES_id,
             year,
             essround,
             denom,
             pray,
             oesch8,
             wgt,
             id,
             country2,
             lrecon,
             galtan,
             spendvtax,
             deregulation,
             redistribution,
             civlib_laworder,
             sociallifestyle,
             religious_principles,
             immigration))


ESS_work_west %<>% as.data.frame()

# The information about religious denomination appears to be incorrect or misleading 
# in France until 2005 and in the United Kingdom from 2004 to 2007 and after 2020.

options(contrasts=c("contr.sum","contr.sum"))


ESS_work_west %<>% within({
    Country <- recode(as.character(country2),
                      c("Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels") -> "Belgium",
                      c("Germany (West)","Germany/Bavaria") -> "Germany",
                      c("Spain/Castilia","Spain/Catalunia","Spain/Basque") -> "Spain",
                      c("United Kingdom/England","United Kingdom/Scotland","United Kingdom/Wales") 
                          -> "United Kingdom",
                      copy=TRUE)
    Country <- factor(Country,levels=sort(levels(Country)))
    denom[Country=="France" & year <= 2005 | 
         Country=="United Kingdom" & year >= 2004 & year <= 2007] <- NA
    cathprot <- recode(as.numeric(denom),
                      "Catholic"     <- 1,
                      "Protestant"   <- 2
                     )
})


ESS_work_west <- merge(ESS_work_west, perc.denom.country.wide,
                       by.x = 'Country',
                       by.y = 'country',
                       all.x = TRUE,
                       all.y = FALSE)


ESS_work_west %<>% within({
    foreach(x = c(lrecon,
                  galtan,
                  spendvtax,
                  deregulation,
                  redistribution,
                  civlib_laworder,
                  sociallifestyle,
                  religious_principles,
                  immigration),
            y = c(lrecon.sq,
                  galtan.sq,
                  spendvtax.sq,
                  deregulation.sq,
                  redistribution.sq,
                  civlib_laworder.sq,
                  sociallifestyle.sq,
                  religious_principles.sq,
                  immigration.sq),
            {
                x <- (x-5)/10
                y <- x^2
            })
    oesch8 <- factor(oesch8,levels=levels(oesch8)[c(3,1:2,4:8)]) # Production workers as baseline
    country <- factor(country2) # Drop countries not included in this subset
    time <- (year - 2010)/10
    contrasts(cathprot) <- contr("sum", base = 1)
    contrasts(oesch8) <- contr("sum", base = 1)
    contrasts(pray) <- contr("sum", base = 1)
    No_religion <- (No_religion - 50)/100
    Denom_Het <- 2*(1-(Cath_Prot/100)^2-(1-Cath_Prot/100)^2)
    Cath_Prot <- (Cath_Prot - 50)/100
    })

# ## Model estimation

library(mclogit)


model_soclife_full <- mclogit(wgt|id~sociallifestyle.sq+
                                immigration.sq+
                                lrecon.sq+
                                sociallifestyle/(cathprot)*(pray)*time+
                                sociallifestyle/(cathprot)*(No_religion+Cath_Prot+Denom_Het)+
                                sociallifestyle/oesch8*time+
                                immigration/(oesch8+pray+cathprot)*time+
                                lrecon/(oesch8+pray+cathprot)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )


# ## Wald tests

source("Wald.R")


(Wald_soclife_full <- 
 Wald(model_soclife_full,
     list(~sociallifestyle.sq,
          ~sociallifestyle,
          ~sociallifestyle:time,
          ~sociallifestyle:cathprot,
          ~sociallifestyle:cathprot:time,
          ~sociallifestyle:pray,
          ~sociallifestyle:pray:time,
          ~sociallifestyle:cathprot:pray,
          ~sociallifestyle:cathprot:pray:time,
          ~sociallifestyle:No_religion,
          ~sociallifestyle:cathprot:No_religion,
          ~sociallifestyle:Cath_Prot,
          ~sociallifestyle:cathprot:Cath_Prot,
          ~sociallifestyle:Denom_Het,
          ~sociallifestyle:cathprot:Denom_Het
           )))


Wald_soclife_full %>%
        dimrename(          
        "sociallifestyle.sq" = "Social Lifestyle squared",
        "sociallifestyle" = "Social Lifestyle",
        "sociallifestyle:time" = "$\\times$ time",
        "sociallifestyle:cathprot" = "$\\times$ Catholic/Protestant",
        "sociallifestyle:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        "sociallifestyle:pray" = "$\\times$ freq of prayer",
        "sociallifestyle:pray:time" = "$\\times$ freq of prayer $\\times$ time",
        "sociallifestyle:cathprot:pray" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer",
        "sociallifestyle:cathprot:pray:time" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer $\\times$ time",
        "sociallifestyle:No_religion" = "$\\times$ secular",
        "sociallifestyle:cathprot:No_religion" = "$\\times$ Catholic/Protestant $\\times$ secular",
        "sociallifestyle:Cath_Prot" = "$\\times$ denom. balance",
        "sociallifestyle:cathprot:Cath_Prot" = "$\\times$ Catholic/Protestant $\\times$ denom. balance",
        "sociallifestyle:Denom_Het" = "$\\times$ denom. heterogeneity",
        "sociallifestyle:cathprot:Denom_Het" = "$\\times$ Catholic/Protestant $\\times$ denom. heterogeneity"
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-soclife/Wald-cathprot-context-soclife-full.tex")

# ## Reduced model

model_soclife <- mclogit(wgt|id~sociallifestyle.sq+
                                immigration.sq+
                                lrecon.sq+
                                sociallifestyle/(cathprot)*(pray)*time+
                                sociallifestyle/(cathprot)*(Cath_Prot+Denom_Het)+
                                sociallifestyle/oesch8*time+
                                immigration/(oesch8+pray+cathprot)*time+
                                lrecon/(oesch8+pray+cathprot)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )



(Wald_soclife <- 
 Wald(model_soclife,
     list(~sociallifestyle.sq,
          ~sociallifestyle,
          ~sociallifestyle:time,
          ~sociallifestyle:cathprot,
          ~sociallifestyle:cathprot:time,
          ~sociallifestyle:pray,
          ~sociallifestyle:pray:time,
          ~sociallifestyle:cathprot:pray,
          ~sociallifestyle:cathprot:pray:time,
          ~sociallifestyle:Cath_Prot,
          ~sociallifestyle:cathprot:Cath_Prot,
          ~sociallifestyle:Denom_Het,
          ~sociallifestyle:cathprot:Denom_Het
           )))


Wald_soclife %>%
        dimrename(          
        "sociallifestyle.sq" = "Social Lifestyle squared",
        "sociallifestyle" = "Social Lifestyle",
        "sociallifestyle:time" = "$\\times$ time",
        "sociallifestyle:cathprot" = "$\\times$ Catholic/Protestant",
        "sociallifestyle:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        "sociallifestyle:pray" = "$\\times$ freq of prayer",
        "sociallifestyle:pray:time" = "$\\times$ freq of prayer $\\times$ time",
        "sociallifestyle:cathprot:pray" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer",
        "sociallifestyle:cathprot:pray:time" = "$\\times$ Catholic/Protestant $\\times$ freq of prayer $\\times$ time",
        "sociallifestyle:Cath_Prot" = "$\\times$ denom. balance",
        "sociallifestyle:cathprot:Cath_Prot" = "$\\times$ Catholic/Protestant $\\times$ denom. balance",
        "sociallifestyle:Denom_Het" = "$\\times$ denom. heterogeneity",
        "sociallifestyle:cathprot:Denom_Het" = "$\\times$ Catholic/Protestant $\\times$ denom. heterogeneity"
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-soclife/Wald-cathprot-context-soclife.tex")


(Wald_soclife_short <- 
 Wald(model_soclife,
     list(~sociallifestyle.sq,
          ~sociallifestyle,
          ~sociallifestyle:time,
          ~sociallifestyle:cathprot,
          ~sociallifestyle:cathprot:time
           )))


Wald_soclife_short %>%
        dimrename(          
        "sociallifestyle.sq" = "Social Lifestyle squared",
        "sociallifestyle" = "Social Lifestyle",
        "sociallifestyle:time" = "$\\times$ time",
        "sociallifestyle:cathprot" = "$\\times$ Catholic/Protestant",
        "sociallifestyle:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="cathprot-context-soclife/Wald-cathprot-context-soclife-short.tex")


save(model_soclife,model_soclife_full,file="model-cathprot-prayer-soclife-context.RData")


save(model_soclife,file="model-churchat-cathprot-soclife-context.RData")

